!-------------------------------------------------------------------------------
! NAME:
!   gps_bufr_decoder
!
! FUNCTION:
!   Read a GPS BUFR file and write the needed data to a comma separated 
!   value (CSV) file.
!
! DESCRIPTION:
!
! REFERENCE:
!
! CALLING SEQUENCE:
!   gps_bufr_decoder.exe <gps_bufr_filename> <path_to_bufr_tables>
!
! INPUTS:
!   gps_bufr_filename = name of GPS TPW BUFR file
!   path_to_bufr_tables = directory path to the BUFR tables
!
! OUTPUTS:
!   Comma separated value (csv) file with this naming convention:
!      (uses the input file name)
!      NWS_<gps_bufr_filename>.csv 
!
! DEPENDENCIES:
!   BUFR lib (BUFRLIB_v10-2-3)
!   bufrtab.TableB_LOC_0_7_1
!   bufrtab.TableB_STD_0_13
!   bufrtab.TableD_LOC_0_7_1
!   bufrtab.TableD_STD_0_13
!
! RESTRICTIONS:
!
! HISTORY:
!   2019-09-01 Peter Keehn  GAMA1  Initial development.
!   2023-01-24 Peter Keehn  GAMA1  Change output format 445.
!
!-------------------------------------------------------------------------------
program gps_bufr_decoder

   implicit none

   character(len=200) :: bufr_filename
   character(len=200) :: out_filename
   character(len=100) :: cbuff
   character(len=100) :: bufr_data_names
   character(len=8) :: station
   character(len=200) :: fname
   character(len=200) :: path_to_bufr_tables
   integer :: ferr
   integer :: ferr2
   integer :: datetime
   integer :: nlv
   integer :: i
   integer :: yyyy
   integer :: yy
   integer :: month
   integer :: day
   integer :: hour
   integer :: minute
   integer :: counter
   integer :: last_slash
   real :: lat
   real :: lon
   real :: elev
   real :: pres
   real :: temp
   real :: rh
   real :: tpw
   real*8, dimension(13,1) :: my_array
   logical :: isexist
   logical :: isopen
   logical :: isnamed

   equivalence (my_array(1,1), station)

   integer ireadmg
   integer ireadsb

   bufr_data_names = "STSN YEAR MNTH DAYS HOUR MINU CLATH CLONH SELV PRES TMDBST REHU TPWT "

   !
   ! Get command line inputs.
   !
   CALL get_command_argument(1, bufr_filename, STATUS=ferr)
   CALL get_command_argument(2, path_to_bufr_tables, STATUS=ferr2)
   if ((ferr+ferr2) /= 0) then
      print *,"Usage: gps_bufr_decoder.exe <gps_bufr_filename> <path_to_bufr_tables>"
      print *,"        gps_bufr_filename = name of GPS TPW BUFR file"
      print *,"      path_to_bufr_tables = directory path to the BUFR tables"
      print *,""
      stop
   end if

   !
   ! Open the BUFR file.
   !
   print *,""
   print "(a15,a)"," in filename: ",trim(bufr_filename)
   open(31,file=bufr_filename,status="old",iostat=ferr)
   if (ferr /= 0) then
      print *," ERROR: open failed for BUFR file: ",trim(bufr_filename)
      stop
   end if

   !
   ! Open the output file.
   !
   last_slash = index(bufr_filename,"/",back=.true.)
   out_filename = "NWS_"//trim(bufr_filename(last_slash+1:))//".csv"
   open(22,file=out_filename,form="formatted",status="replace",iostat=ferr)
   if (ferr /= 0) then
      print *," ERROR: open failed for output file: ",trim(out_filename)
      stop
   end if
   print "(a15,a)","out filename: ",trim(out_filename)
   !
   ! Write the output file header.
   !
   write(22,"(a)") "Ground Based GPS Integrated Precipitable Water Vapor"

   !
   ! Set up BUFR LIB stuff.
   !
   call openbf(31,"SEC3",31)
   !call mtinfo("/data/data311/pkeehn/DPEAS_2/gps_decoder", 90, 91)
   call mtinfo(trim(path_to_bufr_tables), 90, 91)
   call datelen(10)

   cbuff = ""
   counter = 0
   !
   ! Process each message.
   !
   do while (ireadmg(31,cbuff,datetime) == 0)
      counter = counter + 1
      print "(a15,i3)","msg #: ",counter
      print "(a15,i10)","datetime: ",datetime
      print "(a15,a)","mnemonic: ",trim(cbuff)

      !
      ! Process each sub-message.
      !
      do while (ireadsb(31) == 0)
         call ufbint(31, my_array, 13, 1, nlv, trim(bufr_data_names))
         !print *,"nlv: ",nlv
         !
         ! Check for missing values and replace with -999.0
         !
         where (my_array == 10.0e10) my_array = -999d0

         yyyy = my_array(2,1)
         yy = yyyy - (yyyy/100)*100
         month = my_array(3,1)
         day = my_array(4,1)
         hour = my_array(5,1)
         minute = my_array(6,1)
         lat = my_array(7,1)
         lon = my_array(8,1)
         elev = my_array(9,1)
         pres = my_array(10,1)/100d0    !convert from Pa to hPa
         temp = my_array(11,1) - 273.15 !convert to Celsius
         rh = my_array(12,1)
         tpw = my_array(13,1)

         !
         ! Write the data to the output file like this...
         ! 
         ! Ground Based GPS Integrated Precipitable Water Vapor
         ! BBY7, 19/05/03 11:55:00,   7.9, 1011.0,   -1.4, -99,  38.3191, -123.0720,   18.0
         ! ...
         ! ...
         ! ...
         !

         !print 444, station,yyyy,month,day,hour,minute,tpw,pres,temp,rh,lat,lon,elev
         !write(22,444) station,yy,month,day,hour,minute,tpw,pres,temp,rh,lat,lon,elev
         write(22,445) station,station,yy,month,day,hour,minute,tpw,pres,temp,rh,lat,lon,elev
         444 format (a4,", ",i2.2,"/",i2.2,"/",i2.2," ",i2.2,":",i2.2,":","00",7(", ",f0.4))
         445 format (a4,", ",a4,", ",i2.2,"/",i2.2,"/",i2.2," ",i2.2,":",i2.2,":","00",7(", ",f0.4))
      end do
   end do

   call closbf(31)
   close(22)

   !fname = ""
   !inquire(unit=31, exist=isexist, opened=isopen, named=isnamed, name=fname)
   !print *," 31: ",isexist,isopen,isnamed,trim(fname)

   !fname = ""
   !inquire(unit=90, exist=isexist, opened=isopen, named=isnamed, name=fname)
   !print *," 90: ",isexist,isopen,isnamed,trim(fname)

   !fname = ""
   !inquire(unit=91, exist=isexist, opened=isopen, named=isnamed, name=fname)
   !print *," 91: ",isexist,isopen,isnamed,trim(fname)

end program gps_bufr_decoder
