#!/usr/bin/python

"""
copy_dir_files_to_newdir.py

  - copies all the files of a directory to a different directory
  - if file is an HDF file, copies the HDF file, but also converts
    it to NetCDF-4 at the destination directory (using CW-Utilities)
  - useful for copying all the files in an ECN storage directory
    to the STAR web server or STAR ftp server

usage: ./copy_dir_files_to_newdir.py --from_dir [source_dir] --to_dir [destination_dir]

Ron Vogel, SMRC for NOAA CoastWatch
Dec 30, 2016
"""

import sys
#import re
import os
import shutil
import commands
import argparse


def export_ncdf(hdf_file_path):

  # function to convert HDF file to NetCDF-4 using CW-Utilities cwexport.
  #
  # input parameter:  HDF filename with directory path

  hdf_file = os.path.basename(hdf_file_path)
  data_dir = os.path.dirname(hdf_file_path)
  
  # create short file name without extension
  i = hdf_file.find('hdf')
  hdf_file_short = hdf_file[0:i]
  
  # add .nc4 to create netcdf filename
  nc_name = hdf_file_short + 'nc4'

  # run cwutils cwexport to convert hdf file to netcdf4
  # - first checks if "data_source" metadata field exists in HDF global
  #   attributes, and if so, does _not_ write DCS metadata
  #   (cwexport --dcs option searches for "satellite" metadata field but
  #   "satellite" is missing when "data_source" is present, so --dcs option
  #   will not work)
  # - if export fails, writes cwutils error messsage
  cmd1 = 'hdatt --name "data_source" ' + hdf_file_path
  (status, output) = commands.getstatusoutput(cmd1) #if no data_source, command
                                                    #fails & status is non-zero
  if status:    # status is true (non-zero), data_source not present
    #sys.stderr.write(output)  # writes hdatt error message
    #print '\n'
    cmd2 = 'cwexport --dcs --cw ' + hdf_file_path + ' ' + data_dir + '/' + nc_name
  else:         # status is false (zero), data_source is present
    cmd2 = 'cwexport --cw ' + hdf_file_path + ' ' + data_dir + '/' + nc_name
  (status, output) = commands.getstatusoutput(cmd2)
  if status:
    print 'File failed to convert to netcdf: ' + hdf_file
    sys.stderr.write(output)  # writes cwexport error message
    print '\n'
  

def main():

  # get from_directory and to_directory from the arguments
  parser = argparse.ArgumentParser(description='Copies all the files of one directory to another directory')
  parser.add_argument('--from_dir',help='Directory to copy from',required=True)
  parser.add_argument('--to_dir',help='Directory to copy to',required=True)
  args = parser.parse_args()

  print ("From directory: %s" % args.from_dir )
  print ("To directory: %s" % args.to_dir )

  source_dir = args.from_dir
  destination_dir = args.to_dir

  # determine list of files to copy
  files = os.listdir(source_dir)
  
  for file in files:

    # add path names to files  
    source_path = os.path.join(source_dir,file)
    destination_path = os.path.join(destination_dir,file)

    # copy each file to destination
    shutil.copy(source_path,destination_path)

    # if file is HDF, convert it to NetCDF-4 at the destination directory	
    if file.endswith('hdf'):
      export_ncdf(destination_path)


if __name__ == "__main__":
  main()

